<?php
/**
 * ChillNetHosting Service Provider Class
 *
 * @author Conrad van Straaten <conrad@chillnet.co.za>
 * @copyright Copyright (c) 2010
 */
class ChillnetHostingSmsServiceProvider{
    /**
     * @var object smsClient Object
     * @access private
     */
    private $_smsClient = null;
    
    /**
     * @var string Web Service WSDL URL
     * @access private
     */
    private $_webServiceURL = 'http://sms.chillnethosting.co.za/index.php?r=sms/webservice';
    
   /**
    * @var string SMS API Username
    * @access private
    */
    private $_username = 'yourGatewayUsername';

   /**
    * @var string SMS API Password
    * @access private
    */
    private $_password = 'yourGatewayPassword';
       
   /**
    * @var string Sender Identifier
    *   This is the Sender field that should appear in the message
    *   Max Length of 18 if Only Numeric
    *   Max Length of 11 if Alpha numeric
    *   Please note that a source with a ‘+’ sign is treated as alpha numeric and
    *   not numeric so the normal constraints on an alphanumeric senders apply
    *   to it. (Additional restrictions on this field may be enforced by the SMSC)
    * @access public  
    */
   public $identifier = '33979';
      
    /**
     * Default constructor
     * @param string $username
     * @param string $password
     */ 
    public function __construct($username = '', $password = ''){
        if(!empty($username)) $this->_username = $username;
        if(!empty($password)) $this->_password = $password;
        
        $this->_smsClient = new SoapClient($this->_webServiceURL);
    } 
    
    /**
     * Overwrite the default Web Service URL 
     * @param string $url Url
     */ 
    public function setWebServiceUrl($url){
        if(!empty($url)) $this->_webServiceURL = $url; 
    }
    
    /**
     * Get the current Web Service Url
     */ 
    public function getWebServiceUrl(){
        return $this->_webServiceURL;
    }
    
    /**
     * Get the current SMS credit's available
     * @return integer SMS Credit Count
     */ 
    public function getCredits(){
        try{
            $result = $this->_smsClient->getCredits($this->_username,$this->_password);
            $balance = (int) $result;
        }
        catch(Exception $e){
            throw $e;
        }
        return $balance;
    }
    
    /**
     * Send a new SMS message
     * @param string $mobileNumber Mobile Number to send to in international format
     * @param string $message Message to send
     * @return Boolean true if message was sent successfully 
     */ 
    public function sendMessage($mobileNumber, $message, &$msgID = ''){
        try{
            $mobileNumber = ereg_replace("[^0-9]",'',$mobileNumber);
            //Check if the number started with 0, change it to 27 if it did :-)
            if(substr($mobileNumber,0,1) == '0'){
                  $mobileNumber = '27'.substr($mobileNumber,1);
            }
            $result = $this->_smsClient->sendSms($this->_username,$this->_password,$this->identifier,$mobileNumber,$message);
            $pres = $this->_processResult($result, $msgID);
            if($pres !== true){
                throw new Exception($pres);
            }
        }
        catch(Exception $e){
            throw $e;
        }
        return $msgID;
    }

    /**
     * Process SMS Results
     * @param string $result Results as given back by the gateway
     * @param string $messageID Message ID in oder to track delivery status later
     * @return mixed boolean true if successfull or string containing error
     */
    private function _processResult($result,&$messageID){
        //break up the string to only get the error code
        $tmpstr = explode(':',$result);
        $errCode = $tmpstr[0];
        $errStr = $tmpstr[1];
        /*
         Error Codes: 
            1701:Success, Message Submitted Successfully, In this case you will 
                receive the response 1701|<MESSAGE ID>, The message Id can then 
                be used later to map the delivery reports to this message 
            1702:Invalid URL Error, This means that one of the parameters was not 
                provided or left blank 
            1703:Invalid value in username or password field 
            1705:Invalid Message 
            1706:Invalid Destination 
            1707:Invalid Identifier (Sender) 
            1709:User validation failed / Internal Error 
            1710:Internal Error 
            1025.:Insufficient Credit 
            Please Note:- 
                ? Along with the above errors codes, standard SMPP v3.4 error codes may 
                also be returned where applicable. 
                ? Apart from 1709, Please DO NOT RETRY re-sending the message for 
                any other error code (including SMPP v3.4 Error codes ). 
                1710:Internal Error
        */		
        switch($errCode){
            case "1701" : $messageID = $errStr; return true; break;
            case "1702" : return 'Invalid URL Error'; break;
            case "1703" : return 'Invalid value in username or password field'; break;
            case "1705" : return 'Invalid Message'; break;
            case "1706" : return 'Invalid Destination'; break;
            case "1707" : return 'Invalid Identifier (Sender)'; break;
            case "1709" : return 'User validation failed / Internal Error'; break;
            case "1710" : return 'Internal Error'; break;
            case "1025" : return 'Insufficient Credit'; break;
            default : return false; 
        }
    }
    
   /**
    * Set a new Identifier
    *
    * @param integer $identifier Identifier
    */ 
   public function setIdentifier($identifier){
      $this->identifier = $identifier;
   }

   /**
    * Get Delivery Status for a given SMS Message
    *
    * @param string $messageID Unique Message ID returned from the gateway
    * @access public
    */
   public function getDeliveryStatus($messageID){
       try{
            $result = $this->_smsClient->getDeliveryStatus($this->_username,$this->_password,$messageID);
       }catch(Exception $e){
           echo "Something went wrong: ".$e->getMessage();
       }
       return $result;
   }
}